<?php
/**
 * Elementor Pro Module
 *
 * @package HelloPack_Client
 * @since 2.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Elementor Pro module class.
 */
class HelloPack_Module_Elementor_Pro extends HelloPack_Module_Base {

	/**
	 * Minimum required Elementor Pro version
	 *
	 * @var string
	 */
	private $elementor_version_required = '3.9.2';

	/**
	 * Setup module properties
	 */
	protected function setup() {
		$this->id          = 'elementor-pro';
		$this->name        = __( 'Elementor Pro', 'hellopack-client' );
		$this->description = __( 'Manages Elementor Pro updates through centralized updater.', 'hellopack-client' );
		$this->category    = 'plugin';
		$this->version     = '1.0.0';
		$this->target      = array(
			'plugin_dir'  => 'elementor-pro',
			'plugin_file' => 'elementor-pro.php',
		);
	}

	/**
	 * Initialize the module
	 */
	public function init() {
		// Check version compatibility
		if ( ! $this->check_version_compatibility() ) {
			return;
		}

		// Setup Elementor Pro license
		$this->setup_elementor_pro_license();

		// Setup translation API on plugins_loaded
		add_action( 'plugins_loaded', array( $this, 'setup_translation_api' ) );

		// Check for translation updates on WordPress update check
		add_action( 'load-update-core.php', array( $this, 'check_translation_update' ) );
		add_action( 'wp_update_plugins', array( $this, 'check_translation_update' ) );

		// Add translation to update transient
		add_filter( 'site_transient_update_plugins', array( $this, 'add_elementor_translation_to_transient' ), 10 );

		// Load base functionality
		if ( defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			require_once HELLOPACK_CLIENT_PATH . 'inc/modules/elementor/base.php';
		}
	}
	
	/**
	 * Check for translation updates from HelloPack API
	 * This runs only on specific hooks to avoid excessive API calls
	 */
	public function check_translation_update() {
		if ( ! defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			return;
		}

		$locale = get_locale();
		$cache_key = 'hellopack_elementor_pro_translation_' . $locale;

		// Fetch translation metadata from HelloPack API
		$api_url = 'https://hellopack.wp-json.app/api/v1/translations/elementor-pro/' . $locale . '/?id=' . HP_GLOBAL_SERIAL;
		$response = wp_remote_get( $api_url, array( 'timeout' => 10 ) );

		if ( is_wp_error( $response ) ) {
			return;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! isset( $data['success'] ) || ! $data['success'] ) {
			return;
		}

		$translation_info = $data['data'];

		// Get installed translations
		$installed_translations = wp_get_installed_translations( 'plugins' );
		$should_update = false;

		if ( isset( $installed_translations['elementor-pro'][$locale] ) ) {
			$installed_date = $installed_translations['elementor-pro'][$locale]['PO-Revision-Date'];
			$server_date = $translation_info['po_revision_date'];

			// Compare dates - only update if server version is newer
			if ( strtotime( $server_date ) > strtotime( $installed_date ) ) {
				$should_update = true;
			}
		} else {
			// No translation installed yet
			$should_update = true;
		}

		// Store result in transient (12 hours cache)
		if ( $should_update ) {
			set_transient( $cache_key, $translation_info, 12 * HOUR_IN_SECONDS );
		} else {
			delete_transient( $cache_key );
		}
	}

	/**
	 * Add Elementor Pro translation to update transient
	 *
	 * @param object $transient Update transient object.
	 * @return object Modified transient.
	 */
	public function add_elementor_translation_to_transient( $transient ) {
		if ( ! is_object( $transient ) ) {
			$transient = new stdClass();
		}

		if ( ! defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			return $transient;
		}

		$locale = get_locale();
		$cache_key = 'hellopack_elementor_pro_translation_' . $locale;

		// Get cached translation info
		$translation_info = get_transient( $cache_key );

		// Only add to transient if we have cached update info
		if ( $translation_info ) {
			// Initialize translations array if not exists
			if ( ! isset( $transient->translations ) ) {
				$transient->translations = array();
			}

			// Remove existing Elementor Pro translations
			$transient->translations = array_filter(
				$transient->translations,
				function( $translation ) {
					return ! ( isset( $translation['slug'] ) && $translation['slug'] === 'elementor-pro' );
				}
			);

			// Add our custom translation
			$transient->translations[] = array(
				'type'       => 'plugin',
				'slug'       => 'elementor-pro',
				'language'   => $locale,
				'version'    => $translation_info['version'],
				'updated'    => $translation_info['updated'],
				'package'    => $translation_info['package'],
				'autoupdate' => true,
			);
		}

		return $transient;
	}

	/**
	 * Check if Elementor Pro version is compatible
	 *
	 * @return bool
	 */
	private function check_version_compatibility() {
		if ( defined( 'ELEMENTOR_PRO_VERSION' ) && ! version_compare( ELEMENTOR_PRO_VERSION, $this->elementor_version_required, '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'show_version_warning' ) );
			
			if ( get_option( 'elementor_pro_license_key' ) ) {
				delete_option( 'elementor_pro_license_key' );
			}
			
			return false;
		}
		
		return true;
	}

	/**
	 * Show version compatibility warning
	 */
	public function show_version_warning() {
		?>
		<div class="notice notice-error" style="background-color: #fcd0d8">
			<p>
				<?php 
				printf(
					__( '<strong>The current Elementor PRO version on the site is outdated</strong>, and it is not compatible with HelloPack. Please upgrade to at least Elementor PRO %s. This is highly recommended for technical and security reasons.', 'hellopack-client' ),
					$this->elementor_version_required
				); 
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Setup Elementor Pro license
	 */
	private function setup_elementor_pro_license() {
		if ( ! defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			// Clean up if Elementor Pro is not active
			if ( get_option( 'elementor_pro_license_key' ) ) {
				delete_option( 'elementor_pro_license_key' );
			}
			return;
		}

		// Get HelloPack settings
		$hellopack_settings = $this->get_hellopack_settings();
		if ( ! $hellopack_settings ) {
			return;
		}

		// Clean up API request locks
		$this->clean_api_locks();

		// Get license token
		$license_token = $this->get_hello_data( 'token', $hellopack_settings );
		
		// Sync license keys
		$this->sync_license_keys( $license_token );
	}

	/**
	 * Get HelloPack settings
	 *
	 * @return array|false
	 */
	private function get_hellopack_settings() {
		if ( HELLOPACK_CLIENT_NETWORK_ACTIVATED ) {
			return get_site_option( 'hellopack_client' );
		}
		return get_option( 'hellopack_client' );
	}

	/**
	 * Clean API request locks - Always delete to prevent locking
	 */
	private function clean_api_locks() {
		// Always delete the lock option if it exists
		// This prevents the "Request Locked" status from appearing
		delete_option( '_elementor_pro_api_requests_lock' );
	}

	/**
	 * Sync license keys
	 *
	 * @param string $license_token License token.
	 */
	private function sync_license_keys( $license_token ) {
		if ( get_option( 'elementor_pro_license_key' ) != get_option( 'elementor_hellopack_license_key' ) ) {
			delete_option( 'elementor_pro_license_key' );
		}

		if ( ! get_option( 'elementor_hellopack_license_key' ) ) {
			delete_option( 'elementor_pro_license_key' );
			delete_option( '_elementor_pro_license_data' );
			add_option( 'elementor_hellopack_license_key', $license_token, '', 'yes' );
		}

		if ( ! get_option( 'elementor_pro_license_key' ) ) {
			add_option( 'elementor_pro_license_key', $license_token, '', 'yes' );
		}
	}

	/**
	 * Setup translation API
	 */
	public function setup_translation_api() {
		if ( class_exists( 'HPack_Set_API_Servers' ) ) {
			$translation = new HPack_Set_API_Servers();
			$locale = get_locale();
			$translation->set_api_servers(
				'plugin-downloads.elementor.com/v2/translation/',
				'hellopack.wp-json.app/languages/elementor-pro-' . $locale . '.zip?id='
			);
			$translation->init();
		}
	}

	/**
	 * Get data from array recursively
	 *
	 * @param string $key Key to search for.
	 * @param array  $arr Array to search in.
	 * @return mixed
	 */
	private function get_hello_data( $key, array $arr ) {
		$val = array();
		array_walk_recursive(
			$arr,
			function ( $v, $k ) use ( $key, &$val ) {
				if ( $k == $key ) {
					array_push( $val, $v );
				}
			}
		);
		return count( $val ) > 1 ? $val : array_pop( $val );
	}
}